//===========================================================================
// InformEditCtrl.cpp : implementation of the CInformEditCtrl class
//===========================================================================

#include "stdafx.h"
#include "InformEditCtrl.h"
#include "InformStyleRecord.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


//***************************************************************************
// Local definitions
//***************************************************************************

#define MAX_LINE_LENGTH	1020


/////////////////////////////////////////////////////////////////////////////
// StreamText
//
// This structure only exists so that StreamInCallback() doesn't have to 
// call strlen() every time to find out how much text is left.
//
struct StreamText
{
	const char *text;
	LONG		length;
};


//***************************************************************************
// CInformEditCtrl basics
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// Static class member data
//
const LPCTSTR CInformEditCtrl::s_RTFParText = "\\par ";
const LPCTSTR CInformEditCtrl::s_RTFCloseText = "\n\\par }\n.";


/////////////////////////////////////////////////////////////////////////////
// Construction
//
CInformEditCtrl::CInformEditCtrl() :
	m_Facename("Courier New"),
	m_FontSize(100),
	m_LastLineCount(0),
	m_EditLine(-1)
{
}


/////////////////////////////////////////////////////////////////////////////
// Destruction
//
CInformEditCtrl::~CInformEditCtrl()
{
}


/////////////////////////////////////////////////////////////////////////////
// Message Map
//
BEGIN_MESSAGE_MAP(CInformEditCtrl, CRichEditCtrl)
	//{{AFX_MSG_MAP(CInformEditCtrl)
	ON_WM_CREATE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


//***************************************************************************
// CInformEditCtrl diagnostics
//***************************************************************************

#ifdef _DEBUG
void CInformEditCtrl::AssertValid() const
{
	CRichEditCtrl::AssertValid();
}

void CInformEditCtrl::Dump(CDumpContext& dc) const
{
	CRichEditCtrl::Dump(dc);
}
#endif //_DEBUG


//***************************************************************************
// CInformEditCtrl helpers
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// SetText()
//
// Set the text of the edit control, then go through syntax styling it.
// The initial state of each line is stored in a DWORD array so that we can 
// update individual lines when it gets edited later (that code isn't 
// written yet).
//
// My first attempt at this was much more readable, but unfortunately it was 
// also very slow due to the use of SetSelectionCharFormat().  To speed it 
// up I now build up an RTF representation of the text and inser it bodily, 
// but this makes it harder to document and read.
//
void CInformEditCtrl::SetText(LPCTSTR text)
{
	InformStyler fsm;
	InformStyleRecordSet& records = InformStyleRecordSet::GetMasterStyles();
	InformStyle key = InvalidStyle;
	InformStyleRecord * entry;

	// Set up basic style strings:
	int id;
	for ( id = 0 ; NUM_SYNTAX_STYLES > id ; id++ )
		m_RTFStyles[id].Format("\\plain\\f2\\fs%d", m_FontSize/5);

	// Set up quick-access style strings:
	CStringArray rtfColours;
	int colourIndex;
	POSITION pos = records.GetStartPosition();
	while ( NULL != pos )
	{
		records.GetNextAssoc(pos, key, entry);
		ASSERT(entry);
		id = entry->Id();
		ASSERT(NUM_SYNTAX_STYLES > id);
		COLORREF colour = entry->Colour();
		CString rtfColour;
		rtfColour.Format("\\red%d\\green%d\\blue%d;", (int)GetRValue(colour), 
			(int)GetGValue(colour), (int)GetBValue(colour) );

		// Got this colour yet?
		for ( colourIndex = 0 ; rtfColours.GetSize() > colourIndex ; colourIndex++ )
		{
			if ( rtfColours[colourIndex] == rtfColour )
				break;
		}
		if ( rtfColours.GetSize() == colourIndex )
			rtfColours.Add(rtfColour);

		rtfColour.Format("\\cf%d", colourIndex);	// Convert to index code
		m_RTFStyles[id] += rtfColour;
		if ( entry->Bold() )
			m_RTFStyles[id] += "\\b";
		if ( entry->Italic() )
			m_RTFStyles[id] += "\\i";
		m_RTFStyles[id] += " ";
	}

	// Create header at start of working string:
	CString rtfText("{\\rtf1\\ansi{\\fonttbl{\\f2\\fmodern ");
	rtfText += m_Facename;
	rtfText += ";}}\n{\\colortbl";
	for ( colourIndex = 0 ; rtfColours.GetSize() > colourIndex ; colourIndex++ )
		rtfText += rtfColours[colourIndex];
	rtfText += "}\n\\pard";
	m_RTFHeaderText = rtfText;

	// Store initial style:
	m_Styles.RemoveAll();
	m_Styles.Add(fsm.GetState());

	// Perform syntax styling line-by-line:
	int rtfPos = rtfText.GetLength();
	static const LPCTSTR s_RTFParText = "\\par ";
	static const LPCTSTR s_RTFCloseText = "\n\\par }\n.";
	int rtfParLength = strlen(s_RTFParText);
	int rtfParCloseLength = rtfParLength + strlen(s_RTFCloseText);
	int lengthRemaining = strlen(text);
	int rtfIncrement = max(lengthRemaining, 1000);
	int rtfSize = 4*rtfIncrement + rtfParCloseLength + rtfPos + 1;
	LPTSTR rtfBuffer = rtfText.GetBufferSetLength(rtfSize);
	ASSERT(rtfBuffer);
	for ( int line=0 ; 0 < lengthRemaining ; line++ )
	{
		unsigned char styleMap[MAX_LINE_LENGTH+1];
		int lineLength = strcspn(text, "\n");
		if ( MAX_LINE_LENGTH <= lineLength )
		{
			CString message;
			message.Format("The file contained a line longer "
						   "than the maximum allowed (%ld)!",
						   (long)MAX_LINE_LENGTH);
			MessageBox(message, "File Error", MB_ICONEXCLAMATION | MB_OK);
			break;
		}
		CString lineText(text, lineLength++);
		lineText += '\n';
		lengthRemaining -= lineLength;

		ASSERT(m_Styles.GetSize() > line);
		m_Styles.Add(fsm.ColourString(lineText, styleMap));

		char rtfLineText[32*MAX_LINE_LENGTH+1];
		int rtfLinePos = 0;
		int stylePos = 0;
		do
		{
			unsigned char style = styleMap[stylePos];
			ASSERT( NUM_SYNTAX_STYLES > style );

			strcpy(&rtfLineText[rtfLinePos], m_RTFStyles[style]);
			rtfLinePos += m_RTFStyles[style].GetLength();
			do
			{
				char c = *text++;
				if ( '\\' == c || '{' == c || '}' == c )
					rtfLineText[rtfLinePos++] = '\\';
				rtfLineText[rtfLinePos++] = c;
				lineLength--;
			}
			while ( styleMap[++stylePos] == style || ( lineLength && isspace(*text) ) );
		}
		while ( 0 < lineLength );
		rtfLineText[rtfLinePos] = '\0';
		if ( rtfSize <= rtfPos + rtfLinePos + rtfParCloseLength )
		{
			rtfText.ReleaseBuffer();
			rtfSize += max(rtfIncrement, rtfLinePos + rtfParCloseLength);
			rtfBuffer = rtfText.GetBufferSetLength(rtfSize);
			ASSERT(rtfBuffer);
		}
		strcpy(&rtfBuffer[rtfPos], rtfLineText);
		rtfPos += rtfLinePos;
		strcpy(&rtfBuffer[rtfPos], s_RTFParText);
		rtfPos += rtfParLength;
	}

	// Close off the RTF:
	strcpy(&rtfBuffer[rtfPos], s_RTFCloseText);
	rtfText.ReleaseBuffer();

	// Now stream it in:
	EDITSTREAM es;
	StreamText st;
	es.dwCookie = (DWORD)&st;
	es.dwError = 0;
	es.pfnCallback = StreamInCallback;
	st.text = rtfText;
	st.length = rtfText.GetLength();
	StreamIn(SF_RTF,es);

	SetSel(0,0);
	m_LastLineCount = GetLineCount();
}


/////////////////////////////////////////////////////////////////////////////
// static StreamInCallback()
//
// Callback method for streaming in RTF.  This uses and updates a StreamText 
// structure holding a pointer to the remaining text and it's length.
//
DWORD CALLBACK CInformEditCtrl::StreamInCallback(DWORD dwCookie, 
												 LPBYTE pBuff, 
												 LONG cb, LONG FAR * pcb)
{
	StreamText * streamText = (StreamText *)dwCookie;
	ASSERT(pBuff);
	ASSERT(cb);
	ASSERT(pcb);
	ASSERT(streamText);
	if ( NULL == streamText || NULL == streamText->text )
		return -1;
	if ( streamText->length > cb )	// More to do after this?
		*pcb = cb;
	else
		*pcb = streamText->length;
	memcpy(pBuff, streamText->text, (size_t)*pcb);
	streamText->text += *pcb;
	streamText->length -= *pcb;
	return 0;
}


//***************************************************************************
// CInformEditCtrl message handlers
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// OnCreate()
//
// Set the styles/options appropriately (using the fact that each ES_ is the 
// same as it's ECO_ counterpart).  Set the font to a non-proportional one 
// (currently this is forced to be "Courier New"; of course it should be 
// user-selectable).
//
int CInformEditCtrl::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CRichEditCtrl::OnCreate(lpCreateStruct) == -1)
		return -1;

	DWORD options = lpCreateStruct ? ECO_SAVESEL | 
		( lpCreateStruct->style & ES_AUTOVSCROLL | ES_AUTOHSCROLL | 
		  ES_NOHIDESEL | ES_READONLY | ES_WANTRETURN ) :
		ECO_AUTOHSCROLL | ECO_AUTOVSCROLL | ECO_SAVESEL | ECO_WANTRETURN;
	SetOptions(ECOOP_SET, options);

	return 0;
}


