/*=========================================================================*\
|* Bases.c
\*=========================================================================*/

#include "BaseSymbols.h"
#include "Gal2CC.h"

#define BASE_SYMBOL_SCALE_X		1.0
#define BASE_SYMBOL_SCALE_Y		1.0
#define BASE_SYMBOL_ROTATE		0

#define BASE_KEY_Y				404.5
#define BASE_KEY_VERTICAL_SEP	4.1
#define BASE_KEY_MAX			11
#define BASE_KEY_COLOR			15

#define BASE_KEY_TEXT_X			503
#define BASE_KEY_TEXT_FONT		"Arial"
#define BASE_KEY_TEXT_HEIGHT	2.5
#define BASE_KEY_TEXT_ANGLE		0
#define BASE_KEY_TEXT_JUSTIFY	3
#define BASE_KEY_TEXT_STYLE		16

#define BASE_KEY_SYMBOL_X		499
#define BASE_KEY_SYMBOL_SCALE_X	1.0
#define BASE_KEY_SYMBOL_SCALE_Y	1.0
#define BASE_KEY_SYMBOL_ROTATE	0

#define BASE_KEY_HORZ_SS_OFFSET	-553
#define BASE_KEY_VERT_SS_OFFSET	-348


/***************************************************************************\
* Data
*
*/
static const BaseSymbolNameMap baseSymbolNames[] =
{
	{ ImperialNavalBase,		"Imperial Naval Base" },
	{ ImperialScoutBase,		"Imperial Scout Base" },
	{ ImperialNavalDepot,		"Imperial Naval Depot" },
	{ ImperialWayStation,		"Imperial Way Station" },
	{ IndependentNavalBase, 	"Independent Naval Base" },
	{ NonImperialMilitaryBase,	"Military Base" },
	{ TlaukhuBase,				"Tlaukhu Base" },
	{ AslanClanBase ,			"Aslan Clan Base" },
	{ VargrNavalBase,			"Vargr Naval Base" },
	{ VargrCorsairBase, 		"Vargr Corsair Base" },
	{ DroyneNavalBase,			"Droyne Naval Base" },
	{ DroyneGarrison,			"Droyne Garrison" },
	{ HiverNavalBase,			"Hiver Naval Base" },
	{ HiverEmbassyCentre,		"Hiver Embassy Centre" },
	{ HiverTemporaryBase,		"Hiver Temporary Base" },
	{ HiverStagingBase, 		"Hiver Staging Base" },
	{ KKreeNavalBase,			"K'Kree Naval Base" },
	{ KKreeNavalOutpost,		"K'Kree Naval Outpost" },
	{ InterfaceWorld,			"Interface World" },
	{ TerminusWorld,			"Terminus World" },
	{ ZhodaniNavalBase, 		"Zhodani Base" },
	{ ZhodaniRelayStation,		"Zhodani Relay Station" },
	{ ZhodaniNavalDepot,		"Zhodani Naval Depot" },
	{ ExileCamp,				"Exile Camp" },
	{ PrisonCamp,				"Prison Camp" },
	{ ResearchStation,			"Research Station" },
	{ Reserve,					"Reserve" },
	{ 0,						NULL }
};


static const BaseCodeMap standardBaseCodeMap[] =
{
	{ 'A', ImperialNavalBase | ImperialScoutBase },
	{ 'B', ImperialNavalBase | ImperialWayStation },
	{ 'C', VargrCorsairBase },
	{ 'D', ImperialNavalDepot },
	{ 'E', HiverEmbassyCentre },
	{ 'F', IndependentNavalBase | NonImperialMilitaryBase },
	{ 'G', VargrNavalBase },
	{ 'H', VargrCorsairBase | VargrNavalBase },
	{ 'H', HiverStagingBase },
	{ 'I', InterfaceWorld },
	{ 'J', IndependentNavalBase },
	{ 'K', KKreeNavalBase },
	{ 'L', HiverNavalBase },
	{ 'M', NonImperialMilitaryBase },
	{ 'N', ImperialNavalBase },
	{ 'O', KKreeNavalOutpost },
	{ 'P', DroyneNavalBase },
	{ 'Q', DroyneGarrison },
	{ 'R', AslanClanBase },
	{ 'S', ImperialScoutBase },
	{ 'T', TlaukhuBase },
	{ 'T', TerminusWorld },
	{ 'T', HiverTemporaryBase },
	{ 'U', AslanClanBase | TlaukhuBase },
	{ 'W', ImperialWayStation },
	{ 'X', ZhodaniRelayStation },
	{ 'Y', ZhodaniNavalDepot },
	{ 'Z', ZhodaniNavalBase },
	{ '\0', 0 }
};

static const DoubleCoord singleOffsets[1] = 
{
	{ -4.0,  0.0 }
};
static const DoubleCoord doubleOffsets[2] = 
{
	{ -3.5, -2.5 },
	{ -3.5, +2.5 }
};
static const DoubleCoord tripleOffsets[3] = 
{
	{ -3.5, -2.5 },
	{ -4.0,  0.0 },
	{ -3.5, +2.5 }
};
static const DoubleCoord quadrupleOffsets[4] = 
{
	{ +3.5, -2.5 },
	{ +3.5, +2.5 },
	{ -3.5, -2.5 },
	{ -3.5, +2.5 }
};
static const DoubleCoord quintupleOffsets[5] = 
{
	{ +3.5, -2.5 },
	{ +3.5, +2.5 },
	{ -3.5, -2.5 },
	{ -4.0,  0.0 },
	{ -3.5, +2.5 }
};
static const DoubleCoord * baseOffsets[5] =
{
	singleOffsets, doubleOffsets, tripleOffsets, 
	quadrupleOffsets, quintupleOffsets
};

static long sectorBases;
static const char * sectorBaseExtra = NULL;
static BOOL basePrintingMode = TRUE;


/***************************************************************************\
* CountBits()
*
*/
int CountBits(unsigned long mask)
{
	int count;
	for ( count = 0 ; mask ; mask >>= 1 )
	{
		if ( 1 & mask )
			count++;
	}
	return count;
}


/***************************************************************************\
* LookupBaseName()
*
*/
const char * LookupBaseName(unsigned long mask)
{
	int i;
	for ( i = 0 ; baseSymbolNames[i].base ; i++ )
	{
		if ( baseSymbolNames[i].base & mask )
			return baseSymbolNames[i].name;
	}
	return NULL;
}


/***************************************************************************\
* SetBasePrinting()
*
*/
void SetBasePrinting(BOOL mode)			{ basePrintingMode = mode; }


/***************************************************************************\
* InitialiseSectorBases()
*
*/
void InitialiseSectorBases(const char * sector)
{
	static const char * galBaseHeader = "\nBases:\n";
	sectorBases = 0;
	if ( sector )
	{
		sectorBaseExtra = strstr(sector, galBaseHeader);
		if ( sectorBaseExtra )
			sectorBaseExtra += strlen(galBaseHeader);
	}
}


/***************************************************************************\
* OutputBases()
*
*/
void OutputBases(const UWPColumnFormat * format, FileRecord output, 
				 const char * uwpLine, DoubleCoord centre)
{
	const char * str;
	unsigned int lineLength;
	int i;
	long mask;
	int notesCount;
	char bases = toupper(uwpLine[format->bases]);
	long extraMask = 0;
	const DoubleCoord * offsets = NULL;

	if ( !basePrintingMode )
		return;
	str = strchr(uwpLine, '\n');
	lineLength = ( NULL == str ) ? strlen(uwpLine) : str - uwpLine;

	if ( format->notes < lineLength )
	{
		const char * str = strstr(&uwpLine[format->notes], "Ex");
		if ( NULL != str && ( (unsigned int)(str-&uwpLine[format->notes]) <
			  format->notes + format->notesWidth ) )
			extraMask |= ExileCamp;
		str = strstr(&uwpLine[format->notes], "Pr");
		if ( NULL != str && ( (unsigned int)(str-&uwpLine[format->notes]) <
			  format->notes + format->notesWidth ) )
			extraMask |= PrisonCamp;
		str = strstr(&uwpLine[format->notes], "Rs");
		if ( NULL != str && ( (unsigned int)(str-&uwpLine[format->notes]) <
			  format->notes + format->notesWidth ) )
			extraMask |= ResearchStation;
	}
	notesCount = CountBits(extraMask);

	if ( format->bases < lineLength && ' ' != uwpLine[format->bases] )
	{
		for ( i = 0 ; baseSymbolNames[i].base ; i++ )
		{
			if ( standardBaseCodeMap[i].code == bases )
			{
				int count = CountBits(standardBaseCodeMap[i].mask);
				if ( 0 == count )
					break;

				if ( 5 < count )
					count = 5;
				if ( 5 < count + notesCount)
					notesCount = 5 - count;
				offsets = baseOffsets[--count + notesCount];

				sectorBases |= standardBaseCodeMap[i].mask;

				for ( mask = 1 ; 0 <= count ; mask <<= 1 )
				{
					if ( mask & standardBaseCodeMap[i].mask )
					{
						const char * baseName = LookupBaseName(mask);
						OutputSymbol(output, baseName, 
									 centre.x + offsets[count+notesCount].x, 
									 centre.y + offsets[count+notesCount].y, 
									 BASE_SYMBOL_SCALE_X, BASE_SYMBOL_SCALE_Y, 
									 BASE_SYMBOL_ROTATE);
						count--;
					}
				}
				break;
			}
		}
	}

	if ( notesCount-- )
	{
		if ( !offsets )
			offsets = baseOffsets[notesCount];
		for ( mask = 1 ; 0 <= notesCount ; mask <<= 1 )
		{
			if ( mask & extraMask )
			{
				const char * baseName = LookupBaseName(mask);
				OutputSymbol(output, baseName, 
							 centre.x + offsets[notesCount].x, 
							 centre.y + offsets[notesCount].y, 
							 BASE_SYMBOL_SCALE_X, BASE_SYMBOL_SCALE_Y, 
							 BASE_SYMBOL_ROTATE);
				notesCount--;
			}
		}
		sectorBases |= extraMask;
	}
}


/***************************************************************************\
* OutputBaseKey()
*
*/
void OutputBaseKey(FileRecord output, BOOL subsectorOnly)
{
	const char * baseName;
	int count = 0;
	long mask = 1;
	double symbolX = BASE_KEY_SYMBOL_X;
	double textX = BASE_KEY_TEXT_X;
	double y = BASE_KEY_Y;

	if ( 0 == sectorBases )
		return;

	if ( subsectorOnly )
	{
		symbolX += BASE_KEY_HORZ_SS_OFFSET;
		textX += BASE_KEY_HORZ_SS_OFFSET;
		y += BASE_KEY_VERT_SS_OFFSET;
	}

	BeginLayer(output, "MAP BORDER");
	for ( count = 0, mask = 1 ; BASE_KEY_MAX > count && mask ; mask <<= 1 )
	{
		if ( mask & sectorBases )
		{
			baseName = LookupBaseName(mask);
			OutputText(output, baseName, 0, textX, y, 
					   BASE_KEY_COLOR, BASE_KEY_TEXT_FONT, 
					   BASE_KEY_TEXT_HEIGHT, BASE_KEY_TEXT_ANGLE, 
					   BASE_KEY_TEXT_JUSTIFY, BASE_KEY_TEXT_STYLE);
			OutputSymbol(output, baseName, symbolX, y, 
						 BASE_KEY_SYMBOL_SCALE_X, BASE_KEY_SYMBOL_SCALE_Y, 
						 BASE_KEY_SYMBOL_ROTATE);
			count++;
			y -= BASE_KEY_VERTICAL_SEP;
		}
	}
	EndLayer(output);
}


