/*=========================================================================*\
|* Gal2CC.c
\*=========================================================================*/

#include "Gal2CC.h"
#include <time.h>


/***************************************************************************\
* DisplayHelp()
*
* Self-explanatory.  Called when command-line arguments are wrong.
*/
static const char * const helpText = 
	"Gal2CC [options] sector1 [sector2...] \n"
	"Version 1.1.  Creates a Campaign Cartographer 2 script 'sectorn.scr' \n"
	"from each Galactic sector dir 'sectorn\\'; see also \"readme.txt\".\n"
	"\nOptions:\n"
	"-d outputdir  writes all output in this directory;\n"
	"-s#           also outputs a subsector script to 'sectorn#.scr',\n"
	"              where # is one or more of a-p (+ or * for all).\n"
	"-x#           sets hex number printing, where # is one of\n"
	"              o    occupied hexes only (default);\n"
	"              u    unoccupied hexes only;\n"
	"              n,-  no hexes;\n"
	"              a,+  all hexes.\n"
	"-y year       adds the given date text to the legend\n"
	"              (put it in quotes if you want spaces).\n"
	"-abgnptz      prints items on the map (all ON by default):\n"
	"              a    allegiance;           b    base symbols;\n"
	"              g    gas giants;           n    world names;\n"
	"              p    ports;                t    trade routes;\n"
	"              z    zones.\n"
	"              You can turn off options using -a- (for example).\n"
	"Note: Changing options only affects files later in the list.\n";

void DisplayHelp()
{
	puts(helpText);
}


/***************************************************************************\
* main()
*
* This basically identifies command-line arguments and calls the relevant 
* functions.
*/
int main(int argc, char *argv[])
{
	int i;
	int processedCount = 0;
	BOOL abort = FALSE;
	BOOL displayedHelp = FALSE;

	ClearErrorCount();
	srand( (unsigned)time(NULL) );

	for ( i = 1 ; !abort && argc > i ; i++ )
	{
		if ( '-' == argv[i][0] || '/' == argv[i][0] )
		{
			switch(tolower(argv[i][1]))
			{
			  case '?':
			  case 'h':
			  {
				  DisplayHelp();
				  displayedHelp = TRUE;
				  break;
			  }
			  case 'a':	{ SetAllegiancePrinting('-' != argv[i][2]); break; }
			  case 'b':	{ SetBasePrinting('-' != argv[i][2]); break; }
			  case 'd':
			  {
				if ( '-' == argv[i][2] )
					SetOutputDir(NULL);
				else if ( '\0' != argv[i][2] )
					SetOutputDir(&argv[i][2]);
				else if ( argc > ++i )
					SetOutputDir(argv[i]);
				else
					abort = TRUE;
				break;
			  }
			  case 'g':	{ SetGasGiantPrinting('-' != argv[i][2]); break; }
			  case 'n':	{ SetNamePrinting('-' != argv[i][2]); break; }
			  case 'p':	{ SetPortPrinting('-' != argv[i][2]); break; }
			  case 's':
			  {
				int j;
				for ( j = 2 ; argv[i][j] ; j++ )
					AddSubsectorOutput(argv[i][j]);
				break;
			  }
			  case 't':	{ SetTradeRoutePrinting('-' != argv[i][2]); break; }
			  case 'x':
			  {
				switch(tolower(argv[i][2]))
				{
				  case '-':
				  case 'n':
					SetHexNumberPrinting(NoHexNumberPrinting);
					break;
				  case '+':
				  case 'a':
				  case 'f':
					SetHexNumberPrinting(FullHexNumberPrinting);
					break;
				  case 'o':
				  case '0':	/* In case people read it wrong! */
					SetHexNumberPrinting(OccupiedHexNumberPrinting);
					break;
				  case 'u':
					SetHexNumberPrinting(UnoccupiedHexNumberPrinting);
					break;
				  default:
					abort = TRUE;
					break;
				}
				break;
			  }
			  case 'y':
			  {
				if ( '-' == argv[i][2] )
					SetYearText(NULL);
				else if ( '\0' != argv[i][2] )
					SetYearText(&argv[i][2]);
				else if ( argc > ++i )
					SetYearText(argv[i]);
				else
					abort = TRUE;
				break;
			  }
			  case 'z':	{ SetZonePrinting('-' != argv[i][2]); break; }
			  default:
			  {
				char opt[2];	/*##JGW##*/
				opt[0] = argv[i][1] ? argv[i][1] : ' ';
				opt[1] = '\0';
				ReportError("Invalid option -%s!\n", opt);
				abort = TRUE;
				break;
			  }
			}
		}
		else
		{
			if ( !ConvertSector(argv[i]) )
				IncrementErrorCount();
			processedCount++;
		}
	}

	if ( abort || 0 == processedCount )
	{
		if ( !displayedHelp )
			DisplayHelp();
		return 0;
	}

	if ( ErrorCount() )
		printf("There were %d errors!\n", ErrorCount());
	return -ErrorCount();
}
