/*=========================================================================*\
|* Gal2CC.h
\*=========================================================================*/

#ifndef GAL2CC_H
#define GAL2CC_H

#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define SUBSECTOR_HEX_WIDTH 	8
#define SUBSECTOR_HEX_HEIGHT	10

#define SECTOR_HEX_WIDTH	(4*SUBSECTOR_HEX_WIDTH)
#define SECTOR_HEX_HEIGHT	(4*SUBSECTOR_HEX_HEIGHT)

#define	PI		3.1415926535
#define	TWO_PI	6.283185307
#define	HALF_PI	1.570796327

#define RadiansToDegrees(angle)		(360 * angle / TWO_PI)


/***************************************************************************\
|* Type definitions
\***************************************************************************/

#ifndef __cplusplus
	typedef enum { FALSE, TRUE } BOOL;
#endif

typedef enum
{
	NoHexNumberPrinting,
	UnoccupiedHexNumberPrinting,
	OccupiedHexNumberPrinting,
	FullHexNumberPrinting
} HexNumberPrintingMode;

typedef enum
{
	NoHexLimit,
	HexSubsectorLimit,
	HexSectorLimit
} HexLimitMode;

typedef struct
{
	int x;
	int y;
} IntCoord;

typedef struct
{
	double x;
	double y;
} DoubleCoord;

typedef struct
{
	char *			name;
	FILE *			file;
} FileRecord;

typedef struct
{
	unsigned long	base;
	const char		* name;
} BaseSymbolNameMap;

typedef struct
{
	char			code;
	unsigned long	mask;
} BaseCodeMap;

typedef struct
{
	unsigned int	name;
	unsigned int	nameWidth;

	unsigned int	hex;

	unsigned int	port;
	unsigned int	size;
	unsigned int	atmosphere;
	unsigned int	hydrographics;
	unsigned int	populationExponent;
	unsigned int	government;
	unsigned int	lawLevel;
	unsigned int	techLevel;

	unsigned int	bases;

	unsigned int	resources;
	unsigned int	infrastructure;
	unsigned int	labour;
	unsigned int	culture;

	unsigned int	notes;
	unsigned int	notesWidth;

	unsigned int	zone;
	unsigned int	populationMultiplier;
	unsigned int	planetoidBelts;
	unsigned int	gasGiants;

	unsigned int	allegiance;
	unsigned int	stars;
	unsigned int	starsWidth;
	unsigned int	life;
	unsigned int	resource;
	unsigned int	export;
} UWPColumnFormat;


/***************************************************************************\
|* Function declarations
\***************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/* Bases.c */
extern void InitialiseSectorBases(const char * sector);
extern void OutputBaseKey(FileRecord output, BOOL subsectorOnly);
extern void OutputBases(const UWPColumnFormat * format, FileRecord output, 
						const char * uwpLine, DoubleCoord centre);
extern void SetBasePrinting(BOOL on);

/* *Error.c */
extern void ClearErrorCount();
extern int ErrorCount();
extern void IncrementErrorCount();
extern void ReportError(const char * formatString, const char * string);
extern void ReportWriteError(FileRecord output, const char * formatString);

/* FileIO.c */
extern BOOL CheckFileExists(const char * fileName);
extern char * OpenInputFile(const char * fileName);
extern BOOL OpenOutputFile(const char * givenFileName, FileRecord * output);
extern void CloseInputFile(char * fileData);
extern void CloseOutputFile(FileRecord file);
extern void SetOutputDir(const char * dir);

/* *Memory.c */
extern char * Allocate(size_t size);
extern void Release(void * memory);

/* Output.c */
extern double ArcTangent(DoubleCoord start, DoubleCoord end);
extern BOOL BeginOutput(const char * givenFileName, FileRecord * output);
extern void EndOutput(FileRecord file);
extern void BeginLayer(FileRecord output, const char * layer);
extern void EndLayer(FileRecord output);
extern void OutputHeader(FileRecord output);
extern void OutputArc(FileRecord output, double x, double y, double radius, 
					  double startAngle, double endAngle, int colour, 
					  double lineWidth, const char * lineStyle, 
					  const char * fillStyle);
extern void OutputCircle(FileRecord output, double x, double y, double radius, 
						 int colour, double lineWidth, const char * lineStyle, 
						 const char * fillStyle);
extern void OutputHeader(FileRecord output);
extern void OutputLineWithGap(FileRecord output, DoubleCoord start, 
							  DoubleCoord end, double gapRadius, int colour, 
							  double lineWidth, const char * lineStyle, 
							  const char * fillStyle);
extern void OutputSymbol(FileRecord output, const char * symbol, 
						 double x, double y, double hscale, double vscale, 
						 double rotation);
extern void OutputText(FileRecord output, const char * text, int length, 
					   double x, double y, int colour, const char * font, 
					   double height, double angle, int justify, int style);

/* Sector.c */
extern void AddSubsectorOutput(char subsectorLetter);
extern BOOL ConvertSector(const char * shortSectorFileName);
extern BOOL OutputSectorDetails(const char * sector, int sectorNameLength, 
								FileRecord output);
extern void SetYearText(const char * year);

/* Subsector.c */
extern BOOL AdjustRelativeToSubsector(DoubleCoord * location, 
									  int subsectorIndex);
extern BOOL HexNumberToCoord(IntCoord source, DoubleCoord * target, 
							 BOOL limited);
extern BOOL HexNumberToSubsectorCoord(IntCoord source, DoubleCoord * target);
extern BOOL IsInSubsector(IntCoord hex, int index);
extern BOOL ConvertSubsector(const char * sector, 
							 const char * sectorDrive, 
							 const char * sectorDir, 
							 FileRecord output, int index, 
							 BOOL subsectorOnly);

/* System.c */
extern void SetAllegiancePrinting(BOOL on);
extern void SetGasGiantPrinting(BOOL on);
extern void SetHexNumberPrinting(HexNumberPrintingMode mode);
extern void SetNamePrinting(BOOL on);
extern void SetPortPrinting(BOOL on);
extern void SetZonePrinting(BOOL on);
extern void ClearSystems();
extern void CompleteSystems(FileRecord output, int subsectorIndex, 
							BOOL subsectorOnly);
extern BOOL OutputSystem(const UWPColumnFormat * format, FileRecord output, 
						 const char * uwpLine, int subsectorIndex, 
						 BOOL subsectorOnly);

/* TextManipulation.c */
extern int LineLength(const char * text);
extern BOOL NextLine(char ** text);
extern void StripTrailingWhitespace(char * text);
extern char * NewTextCopy(const char * sourceText);
extern void DeleteTextCopy(char * text);

/* TradeRoute.c */
extern void SetTradeRoutePrinting(BOOL on);
extern BOOL OutputTradeRoute(FileRecord output, const char * tradeLine, 
							 int subsectorIndex, BOOL subsectorOnly);

#ifdef __cplusplus
}
#endif

#endif	/* GAL2CC_H */

