/*=========================================================================*\
|* Subsector.c
\*=========================================================================*/

#include "Gal2CC.h"

/* General */
#define SUBSECTOR_NAME_COLUMN		3
#define SUBSECTOR_FILE_COLUMN		29
#define SUBSECTOR_NAME_MAX_LENGTH	26
#define SUBSECTOR_FILE_MAX_LENGTH	20

#define	HEX_WIDTH					20
#define	HEX_INITIAL_X				10
#define	HEX_HORIZONTAL_SEPARATION	15
#define	HEX_INITIAL_Y				8.66025404
#define	HEX_HEIGHT					17.32050808

/* For the sector map key */
#define SUBSECTOR_KEY_Y 			611
#define SUBSECTOR_KEY_VERTICAL_SEP	4.5
#define SUBSECTOR_KEY_MAX			16

#define SUBSECTOR_KEY_X 			501
#define SUBSECTOR_KEY_COLOR			15
#define SUBSECTOR_KEY_FONT			"Arial"
#define SUBSECTOR_KEY_HEIGHT		3
#define SUBSECTOR_KEY_ANGLE 		0
#define SUBSECTOR_KEY_JUSTIFY		0
#define SUBSECTOR_KEY_STYLE 		16

/* For the subsector map name */
#define SUBSECTOR_NAME_X 			62.5
#define SUBSECTOR_NAME_Y 			-15
#define SUBSECTOR_NAME_COLOR		SUBSECTOR_KEY_COLOR
#define SUBSECTOR_NAME_FONT			SUBSECTOR_KEY_FONT
#define SUBSECTOR_NAME_HEIGHT		5
#define SUBSECTOR_NAME_ANGLE 		SUBSECTOR_KEY_ANGLE
#define SUBSECTOR_NAME_JUSTIFY		7
#define SUBSECTOR_NAME_STYLE 		SUBSECTOR_KEY_STYLE

/* For the subsector map adjacent names */
#define SUBSECTOR_NEIGHBOUR_COLOR		SUBSECTOR_NAME_COLOR
#define SUBSECTOR_NEIGHBOUR_FONT		SUBSECTOR_NAME_FONT
#define SUBSECTOR_NEIGHBOUR_HEIGHT		4
#define SUBSECTOR_NEIGHBOUR_JUSTIFY		4
#define SUBSECTOR_NEIGHBOUR_STYLE 		SUBSECTOR_NAME_STYLE

typedef struct
{
	IntCoord	subsectorOffset;
	DoubleCoord	position;
	int			angle;
} NeighbourInfo;


/***************************************************************************\
* Data
*
*/
static UWPColumnFormat format =
{
	0,	/* name */
	14,	/* nameWidth */

	14,	/* hex */

	19,	/* port */
	20,	/* size */
	21,	/* atmosphere */
	22,	/* hydrographics */
	23,	/* populationExponent */
	24,	/* government */
	25,	/* lawLevel */
	27,	/* techLevel */

	30,	/* bases */

	-1,	/* resources */
	-1,	/* infrastructure */
	-1,	/* labour */
	-1,	/* culture */

	32,	/* notes */
	16,	/* notesWidth */

	48,	/* zone */
	51,	/* populationMultiplier */
	52,	/* planetoidBelts */
	53,	/* gasGiants */

	55,	/* allegiance */
	-1,	/* stars */
	0,	/* starsWidth */
	58,	/* life */
	59,	/* resource */
	60,	/* export */
};

static char subsectorNames[SUBSECTOR_KEY_MAX][SUBSECTOR_NAME_MAX_LENGTH+1] =
{
	"", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""
};

static NeighbourInfo neighbour[4] =
{
	{ {  0, -1}, {  62.5, 186.865 },   0 }, 
	{ { -1,  0}, {  -5.0,  95.933 },  90 }, 
	{ { +1,  0}, { 130.0,  95.933 }, 270 }, 
	{ {  0, +1}, {  62.5,  -5.000 }, 180 }, 
};


/***************************************************************************\
* HexNumberToCoord()
*
*/
BOOL AdjustRelativeToSubsector(DoubleCoord * location, int subsectorIndex)
{
	if ( NULL == location || 0 > subsectorIndex || 16 <= subsectorIndex )
		return FALSE;
	location->x -= SUBSECTOR_HEX_WIDTH * HEX_HORIZONTAL_SEPARATION * 
					(subsectorIndex % 4);
	location->y -= SUBSECTOR_HEX_HEIGHT * HEX_HEIGHT * ((15-subsectorIndex) / 4);
	return TRUE;
}

		
/***************************************************************************\
* HexNumberToCoord()
*
*/
BOOL HexNumberToCoord(IntCoord source, DoubleCoord * target, BOOL limited)
{
	if ( NULL == target )
		return FALSE;
	if ( limited && 
		 ( 0 >= source.x || SECTOR_HEX_WIDTH < source.x || 
		   0 >= source.y || SECTOR_HEX_HEIGHT < source.y ) )
		return FALSE;

	target->x = (double)HEX_INITIAL_X + 
				(double)HEX_HORIZONTAL_SEPARATION * (source.x - 1);
	target->y = (double)HEX_INITIAL_Y + 
				(double)HEX_HEIGHT * (SECTOR_HEX_HEIGHT - source.y);
	if ( 1 & source.x )
		target->y += (double)HEX_HEIGHT / 2;
	return TRUE;
}


/***************************************************************************\
* HexNumberToSubsectorCoord()
*
*/
BOOL HexNumberToSubsectorCoord(IntCoord source, DoubleCoord * target)
{
	if ( NULL == target )
		return FALSE;

	target->x = (double)HEX_INITIAL_X + (double)HEX_HORIZONTAL_SEPARATION * 
					((source.x - 1) % SUBSECTOR_HEX_WIDTH);
	target->y = (double)HEX_INITIAL_Y + (double)HEX_HEIGHT * 
					(SUBSECTOR_HEX_HEIGHT - 1 - 
					 ((source.y - 1) % SUBSECTOR_HEX_HEIGHT));
	if ( 1 & source.x )
		target->y += (double)HEX_HEIGHT / 2;
	return TRUE;
}


/***************************************************************************\
* IsInSubsector()
*
*/
BOOL IsInSubsector(IntCoord hex, int index)
{
	if ( 0 > index || 16 <= index )
		return FALSE;
	return ( (hex.x-1) / SUBSECTOR_HEX_WIDTH == index % 4 ) &&
		   ( (hex.y-1) / SUBSECTOR_HEX_HEIGHT == index / 4 );
}


/***************************************************************************\
* ConvertSubsector()
*
*/
BOOL ConvertSubsector(const char * sector,
					  const char * sectorDrive, const char * sectorDir,
					  FileRecord output, int index, BOOL subsectorOnly)
{
	char drive[_MAX_DRIVE];
	char dir[_MAX_DIR];
	char fname[_MAX_FNAME];
	char ext[_MAX_EXT];
	char subsectorFileName[_MAX_PATH];

	int i;
	char * str;
	const char * constStr;
	const char * subsectorLine;
	char * subsector;
	char subsectorLineID[4];
	BOOL fileExists = TRUE;

	if ( NULL == sector || NULL == sectorDrive || NULL == sectorDir ||
		 0 > index || SUBSECTOR_KEY_MAX <= index )
		return FALSE;

	/* Search for line with subsector details - we need to do this
	   repeatedly to get past the sector neighbours for E: and N:. */
	sprintf(subsectorLineID, "\n%c:", 'A' + index);
	constStr = sector;
	do
	{
		subsectorLine = constStr+1;
		constStr = strstr(constStr+3, subsectorLineID);
	}
	while ( NULL != constStr );
	if ( subsectorLine == sector+1 )
		return FALSE;

	/* Extract name */
	strncpy(subsectorNames[index], &subsectorLine[SUBSECTOR_NAME_COLUMN],
			SUBSECTOR_NAME_MAX_LENGTH);
	subsectorNames[index][SUBSECTOR_NAME_MAX_LENGTH] = '\0';
	str = strchr(subsectorNames[index], '\n');
	if ( NULL != str )
	{
		*str = '\0';
		fileExists = FALSE;
	}
	else
		str = &subsectorNames[index][SUBSECTOR_NAME_MAX_LENGTH];
	while( --str >= subsectorNames[index] )
	{
		if ( !isspace(*str) )
			break;
		*str = '\0';
	}

	/* Output key entry */
	if ( strlen(subsectorNames[index]) )
	{
		BeginLayer(output, "MAP BORDER");
		if ( subsectorOnly )
		{
			char buffer[SUBSECTOR_NAME_MAX_LENGTH+11];
			strcpy(buffer, subsectorNames[index]);
			strcat(buffer, " Subsector");
			OutputText(output, buffer, 0, 
					   SUBSECTOR_NAME_X, SUBSECTOR_NAME_Y, 
					   SUBSECTOR_NAME_COLOR, SUBSECTOR_NAME_FONT, 
					   SUBSECTOR_NAME_HEIGHT, SUBSECTOR_NAME_ANGLE, 
					   SUBSECTOR_NAME_JUSTIFY, SUBSECTOR_NAME_STYLE);
		}
		else
			OutputText(output, subsectorNames[index], 0, 
					   SUBSECTOR_KEY_X, SUBSECTOR_KEY_Y - 
					   (double)SUBSECTOR_KEY_VERTICAL_SEP * index, 
					   SUBSECTOR_KEY_COLOR, SUBSECTOR_KEY_FONT, 
					   SUBSECTOR_KEY_HEIGHT, SUBSECTOR_KEY_ANGLE, 
					   SUBSECTOR_KEY_JUSTIFY, SUBSECTOR_KEY_STYLE);
		EndLayer(output);
	}

	/* Neighbouring subsector names */
	if ( subsectorOnly )
	{
		BeginLayer(output, "MAP BORDER");
		for ( i = 0 ; 4 > i ; i++ )
		{
			int neighbourIndex;
			IntCoord ss;

			ss.x = ( index % 4 ) + neighbour[i].subsectorOffset.x;
			ss.y = ( index / 4 ) + neighbour[i].subsectorOffset.y;
			if ( 0 > ss.x || 4 <= ss.x || 0 > ss.y || 4 <= ss.y )
				continue;
			neighbourIndex = ss.x + 4 * ss.y;
			if ( strlen(subsectorNames[neighbourIndex]) )
				OutputText(output, subsectorNames[neighbourIndex], 0, 
						   neighbour[i].position.x, neighbour[i].position.y, 
						   SUBSECTOR_NEIGHBOUR_COLOR, SUBSECTOR_NEIGHBOUR_FONT, 
						   SUBSECTOR_NEIGHBOUR_HEIGHT, neighbour[i].angle, 
						   SUBSECTOR_NEIGHBOUR_JUSTIFY, SUBSECTOR_NEIGHBOUR_STYLE);
		}
		EndLayer(output);
	}

	/* Handle file */
	if ( fileExists )
	{
		constStr = strchr(&subsectorLine[SUBSECTOR_FILE_COLUMN], '\n');
		if ( constStr )
			i = (int)(constStr - &subsectorLine[SUBSECTOR_FILE_COLUMN]);
		else
			i = _MAX_PATH-1;
		i = min(i, SUBSECTOR_FILE_MAX_LENGTH);
		strncpy(subsectorFileName, &subsectorLine[SUBSECTOR_FILE_COLUMN], i);
		str = &subsectorFileName[i];
		*str = '\0';
		while( --str >= subsectorFileName )
		{
			if ( !isspace(*str) )
				break;
			*str = '\0';
		}
		strcpy(drive, sectorDrive);
		strcpy(dir, sectorDir);
		strcat(dir, "map\\");
		_splitpath(subsectorFileName, NULL, NULL, fname, ext);
		_makepath(subsectorFileName, drive, dir, fname, ext);
		subsector = OpenInputFile(subsectorFileName);
		if ( !subsector )
			return FALSE;

		subsectorLine = subsector;
		while ( *subsectorLine )
		{
			switch (*subsectorLine)
			{
			  case '@':
			  { /* Ignore header */
				break;
			  }
			  case '#':
			  { /* Ignore comment */
				break;
			  }
			  case '$':
			  {
				OutputTradeRoute(output, subsectorLine, index, 
								 subsectorOnly);
				break;
			  }
			  default:
			  { /* Guess this is system info */
				OutputSystem(&format, output, subsectorLine, index,
							 subsectorOnly);
				break;
			  }
			}
			subsectorLine = strchr(subsectorLine, '\n');
			if ( NULL == subsectorLine )
				break;
			subsectorLine++;
		}
		CloseInputFile(subsector);
		return TRUE;
	}
	else
	{
		static char errMsg[] = "No subsector file specified for # (%s)!\n";
		errMsg[strlen(errMsg)-8] = 'A' + index;	/*##JGW##*/
		ReportError(errMsg, subsectorNames[index]);
		return FALSE;
	}
}


